<?php
declare (strict_types = 1);

namespace app\admin\controller;

use think\Request;
use think\facade\View;
use think\facade\Db;
use think\facade\Log;
use think\Response;

class Html extends Common
{
    /**
     * 显示资源列表
     *
     * @return \think\Response
     */
    public function index()
    {
        // 权限拦截：查看权限
        if (!$this->hasPermission('html_view')) {
            if (request()->isAjax() || request()->isPost()) {
                return json(['status'=>'fail','msg'=>'无权限查看html生成']);
            }
            return '<div style="display:flex;align-items:center;justify-content:center;height:80vh;font-size:22px;color:#c00;font-weight:bold;">当前用户无访问html生成权限！</div>';
        }
        return View::fetch('index');
    }

    private function getAllSubCategoryIds($categoryId, $allCategories)
    {
        $ids = [$categoryId];
        foreach ($allCategories as $cat) {
            if ($cat['pid'] == $categoryId) {
                $ids = array_merge($ids, $this->getAllSubCategoryIds($cat['id'], $allCategories));
            }
        }
        return $ids;
    }

    // 获取页面总数
    public function getPageCount()
    {
        $type = input('type', 'all');
        if ($type == 'index') {
            $total = 1;
        } elseif ($type == 'list') {
            $total = $this->getListPagesCount();
        } elseif ($type == 'content') {
            $total = $this->getContentPagesCount();
        } else {
            $total = $this->getAllPagesCount();
        }
        // 读取 last_time
        $lastTime = '--';
        $file = root_path() . 'html/last_generate_time_' . $type . '.txt';
        if (file_exists($file)) {
            $lastTime = file_get_contents($file);
        }
        return json([
            'code' => 1,
            'data' => [
                'total' => $total,
                'last_time' => $lastTime
            ]
        ]);
    }

    private function clearHtmlDir($dir)
    {
        if (!is_dir($dir)) return;
        $files = scandir($dir);
        foreach ($files as $file) {
            if ($file == '.' || $file == '..') continue;
            $path = $dir . DIRECTORY_SEPARATOR . $file;
            if (is_dir($path)) {
                $this->clearHtmlDir($path);
                @rmdir($path);
            } else {
                @unlink($path);
            }
        }
    }

    // 分批生成静态页面
    public function generateStaticBatch()
    {
        try {
            $type = input('type', 'all');
            $batch = (int)input('post.batch', 0);
            $batchSize = (int)input('post.batchSize', 5);

            if ($type == 'all' && $batch == 0) {
                $htmlDir = root_path() . 'html';
                $this->clearHtmlDir($htmlDir);
            }

            $allPages = $this->getAllPages();

            // 按type筛选
            if ($type == 'index') {
                $pages = array_filter($allPages, function($p){ return $p['type']=='index'; });
            } elseif ($type == 'list') {
                $pages = array_filter($allPages, function($p){ return $p['type']=='list' || $p['type']=='list_page'; });
            } elseif ($type == 'content') {
                $pages = array_filter($allPages, function($p){ return $p['type']=='content'; });
            } else {
                $pages = $allPages;
            }
            $pages = array_values($pages);

            $totalPages = count($pages);
            $start = $batch * $batchSize;
            $end = min($start + $batchSize, $totalPages);
            $batchPages = array_slice($pages, $start, $batchSize);

            $staticPath = root_path();
            $successCount = 0;
            $errors = [];
            $processedFiles = [];

            foreach ($batchPages as $page) {
                try {
                    $content = $this->generatePageContent($page);
                    $filePath = $staticPath . ltrim($page['path'], '/');
                    $fileDir = dirname($filePath);
                    if (!is_dir($fileDir)) {
                        mkdir($fileDir, 0755, true);
                    }
                    if (file_put_contents($filePath, $content)) {
                        $successCount++;
                        $processedFiles[] = $page['path'];
                    }
                } catch (\Exception $e) {
                    $errors[] = "生成页面 {$page['path']} 失败: " . $e->getMessage();
                }
            }

            // 最后一批时写入历史时间
            if (($batch + 1) * $batchSize >= $totalPages) {
                $lastTimeFile = root_path() . 'html/last_generate_time_' . $type . '.txt';
                $htmlDir = dirname($lastTimeFile);
                if (!is_dir($htmlDir)) {
                    mkdir($htmlDir, 0755, true);
                }
                file_put_contents($lastTimeFile, date('Y-m-d H:i:s'));
            }

            $this->logAction('批量生成静态页面，类型:' . $type . '，批次:' . $batch);
            return json([
                'code' => 1,
                'msg' => "成功生成 {$successCount} 个页面",
                'data' => [
                    'success' => $successCount,
                    'errors' => $errors,
                    'processedFiles' => $processedFiles,
                    'batch' => $batch,
                    'totalBatches' => ceil($totalPages / $batchSize)
                ]
            ]);
        } catch (\Exception $e) {
            return json(['code' => 0, 'msg' => '生成静态页面失败：' . $e->getMessage()]);
        }
    }

    // 获取所有页面
  
private function getAllPages()
{
    $pages = [];
    $staticRoot = 'html/';

    // 1. 首页
    $pages[] = [
        'type' => 'index',
        'path' => 'index.html',
        'data' => []
    ];

    // 2. 所有栏目
    $allCategories = Db::name('category')->where('status', 1)->select()->toArray();

    foreach ($allCategories as $category) {
        // 用link字段生成目录（去掉首尾斜杠）
        $link = trim($category['link'], '/');
        if (empty($link)) continue;

        $model = Db::name('model')->where('id', $category['model'])->find();
        if (!$model) continue;

        if ($model['type'] == 1) { // 单页
            // 没有内容模板则不生成
            if (empty($category['contenttpl'])) continue;
            $pages[] = [
                'type' => 'about',
                'path' => $staticRoot . $link . '/index.html',
                'data' => [
                    'id' => $category['id']
                ]
            ];
        } else if ($model['type'] == 2) { // 列表
            // 没有列表模板则不生成
            if (empty($category['listtpl'])) continue;
            $pages[] = [
                'type' => 'list',
                'path' => $staticRoot . $link . '/index.html',
                'data' => [
                    'id' => $category['id']
                ]
            ];

            $categoryIds = $this->getAllSubCategoryIds($category['id'], $allCategories);
            $contentsCount = Db::name('content')
                ->where('status', 1)
                ->whereIn('scode', $categoryIds)
                ->count();

            // 自动读取模板num
            $templateFile = root_path() . 'template/index/' . $category['listtpl'];
            $pageSize = 10; // 默认每页10条
            if (is_file($templateFile)) {
                $tplContent = file_get_contents($templateFile);
                if (preg_match('/\{long:list\s+[^}]*num\s*=\s*["\']?(\d+)["\']?/i', $tplContent, $match)) {
                    $pageSize = intval($match[1]);
                }
            }

            $totalPages = ceil($contentsCount / $pageSize);

            for ($page = 2; $page <= $totalPages; $page++) {
                $pages[] = [
                    'type' => 'list_page',
                    'path' => $staticRoot . $link . '_' . $page . '/index.html',
                    'data' => [
                        'id' => $category['id'],
                        'page' => $page
                    ]
                ];
            }

            // 没有内容模板则不生成内容页
            if (empty($category['contenttpl'])) continue;
            $contents = Db::name('content')
                ->where('status', 1)
                ->where('scode', $category['id'])
                ->select()
                ->toArray();

            foreach ($contents as $content) {
                // 直接用内容的link字段生成物理路径
                $contentLink = ltrim($content['link'], '/');
                $pages[] = [
                    'type' => 'content',
                    'path' => $staticRoot . $contentLink,
                    'data' => [
                        'id' => $content['id']
                    ]
                ];
            }
        }
    }

    return $pages;
}

    // 统计所有页面数量
    private function getAllPagesCount() {
        return count($this->getAllPages());
    }
    // 统计栏目页数量
    private function getListPagesCount() {
        return count(array_filter($this->getAllPages(), function($p){ return $p['type']=='list' || $p['type']=='list_page'; }));
    }
    // 统计内容页数量
    private function getContentPagesCount() {
        return count(array_filter($this->getAllPages(), function($p){ return $p['type']=='content'; }));
    }

    // 生成页面内容
    public function generatePageContent($page)
    {
        try {
            $viewConfig = [
                'view_path'   => root_path() . 'template' . DIRECTORY_SEPARATOR,
                'view_suffix' => 'html',
                'view_depr' => DIRECTORY_SEPARATOR,
                'tpl_begin' => '{',
                'tpl_end' => '}',
                'taglib_begin' => '{',
                'taglib_end' => '}',
                'tpl_cache' => false
            ];
            config($viewConfig, 'view');

            $app = app();
            $indexController = new \app\home\controller\Index($app);

            switch ($page['type']) {
                case 'index':
                    $content = $indexController->index();
                    break;
                case 'list':
                    $sort = Db::name('category')->where('id', $page['data']['id'])->find();
                    if (!$sort) throw new \Exception("栏目不存在: {$page['data']['id']}");
                    $indexController->getList($sort);
                    $templateFile = 'index/' . str_replace('.html', '', $sort['listtpl']);
                    $content = View::fetch($templateFile);
                    break;
                case 'list_page':
                    $sort = Db::name('category')->where('id', $page['data']['id'])->find();
                    if (!$sort) throw new \Exception("栏目不存在: {$page['data']['id']}");
                    request()->withGet(['page' => $page['data']['page']]);
                    $indexController->getList($sort);
                    View::assign('current_page', $page['data']['page']);
                    $templateFile = 'index/' . str_replace('.html', '', $sort['listtpl']);
                    $content = View::fetch($templateFile);
                    break;
                case 'about':
                    $sort = Db::name('category')->where('id', $page['data']['id'])->find();
                    if (!$sort) throw new \Exception("栏目不存在: {$page['data']['id']}");
                    $indexController->getAbout($sort);
                    $templateFile = $sort['contenttpl'];
                    if (strpos($templateFile, 'index/') !== 0) {
                        $templateFile = 'index/' . str_replace('.html', '', $templateFile);
                    }
                    $content = View::fetch($templateFile);
                    break;
                case 'content':
                    $contentInfo = Db::name('content')->where('id', $page['data']['id'])->find();
                    if (!$contentInfo) throw new \Exception("内容不存在: {$page['data']['id']}");
                    $sort = Db::name('category')->where('id', $contentInfo['scode'])->find();
                    $indexController->getContent($page['data']['id']);
                    $templateFile = 'index/' . str_replace('.html', '', $sort['contenttpl']);
                    $content = View::fetch($templateFile);
                    break;
                default:
                    throw new \Exception("未知的页面类型: {$page['type']}");
            }

            if (empty($content)) {
                throw new \Exception("页面内容为空");
            }

            return $this->processUrls($content);

        } catch (\Exception $e) {
            throw $e;
        }
    }

    // 处理内容中的URL为绝对路径
    private function processUrls($content)
    {
        $system = Db::name('system')->where('id', 1)->find();
        $domain = isset($system['domain']) && $system['domain'] ? $system['domain'] : request()->domain();

        // 如需替换模板标签可在此添加
        $replace = [
            // '__PUBLIC__' => $domain . '/public',
            // '__STATIC__' => $domain . '/static',
            // '__ROOT__' => $domain,
            // './public' => $domain . '/public',
            // './static' => $domain . '/static',
            // '/uploads/' => $domain . '/uploads/',
        ];
        foreach ($replace as $k => $v) {
            $content = str_replace($k, $v, $content);
        }

        return $content;
    }
}