<?php
declare (strict_types = 1);

namespace app\admin\controller;

use think\Request;
use think\facade\View;
use think\facade\Db;
use think\facade\Session;

class Config extends Common
{
    /**
     * 显示资源列表
     *
     * @return \think\Response
     */
    public function index()
    {

        // 权限拦截：查看权限
        if (!$this->hasPermission('config_view')) {
            if (request()->isAjax() || request()->isPost()) {
                return json(['status'=>'fail','msg'=>'无权限查看配置参数']);
            }
            return '<div style="display:flex;align-items:center;justify-content:center;height:80vh;font-size:22px;color:#c00;font-weight:bold;">当前用户无访问配置参数权限！</div>';
        }

        $config=Db::name('config')->where('id',1)->find();

        if($data=input('post.')){
            if(isset($data['search'])){
                $data['search']=implode(',',$data['search']);
            }else{
                $data['search']='';
            }

            // 权限拦截：编辑权限
            if (!$this->hasPermission('config_edit')) {
                return json(['status'=>'fail','msg'=>'无权限编辑配置参数']);
            }

            // 获取数据库中的原始数据
            $originalData = Db::name('config')->where('id',1)->find();
            // 只记录有变动的字段
            $logArr = [];
            $hasChanges = false;
            foreach ($data as $key => $value) {
                $old = isset($originalData[$key]) ? $originalData[$key] : '';
                $old = is_null($old) ? '' : $old;
                $new = is_null($value) ? '' : $value;
                if ($old != $new) {
                    $hasChanges = true;
                    $logArr[] = ['字段'=>$key, '原'=>$old, '改为'=>$new];
                }
            }
            //结构变化时，更新内容管理下来拉框
            if($data['structure']==$originalData['structure']){
                if($data['structure']=='1'){
                    // 自动更新内容管理下来框
                };
                if($data['structure']=='2'){
                    // 自动更新内容管理下来框
                };
            }
            if (!$hasChanges) {
                return json(['status' => 'fail', 'msg' => '没有做任何修改']);
            }
            // 设置 session 时长，将分钟转换为秒
            if(isset($data['session_time'])){
                $sessionTimeInSeconds=(int)$data['session_time']*60;
                Session::init([
                    'expire'=>$sessionTimeInSeconds,
                ]);
            }
            

            $url_mode=Db::name('config')->where('id', 1)->value('url_mode');
            if($data['url_mode']!==$url_mode){
                //伪静态配置文件
                if($data['url_mode']=='1'){
                    // 生成 .htaccess 文件内容
                    $htaccess = <<<HTA
                    <IfModule mod_rewrite.c>
                    Options +FollowSymlinks -Multiviews
                    RewriteEngine On
                    
                    RewriteCond %{REQUEST_FILENAME} !-d
                    RewriteCond %{REQUEST_FILENAME} !-f
                    RewriteRule ^(.*)$ index.php?/$1 [QSA,PT,L]
                    </IfModule>
                    HTA;
                        // 写入到根目录
                    file_put_contents(root_path() . '.htaccess', $htaccess);
                }
                //静态页配置文件
                if($data['url_mode']=='2'){
                    $htaccess = <<<HTA
                    <IfModule mod_rewrite.c>
                    Options +FollowSymlinks -Multiviews
                    RewriteEngine On

                    # 允许直接访问 admin.php
                    RewriteCond %{REQUEST_URI} ^/admin\.php$ [NC]
                    RewriteRule ^.*$ admin.php [L]

                    # 允许 /admin 路径及其子路径
                    RewriteCond %{REQUEST_URI} ^/admin($|/) [NC]
                    RewriteRule ^.*$ index.php?/$1 [QSA,PT,L]

                    # public目录下的静态文件直接访问
                    RewriteCond %{REQUEST_FILENAME} -f
                    RewriteRule ^public/.*$ - [L]

                    # 处理验证码请求
                    RewriteCond %{REQUEST_URI} ^/captcha [OR]
                    RewriteCond %{REQUEST_URI} ^/admin/login/captcha [OR]
                    RewriteCond %{REQUEST_URI} ^/admin/captcha
                    RewriteRule ^(.*)$ index.php?s=$1 [QSA,PT,L]

                    # 首页静态化
                    RewriteCond %{REQUEST_URI} ^/?$ [OR]
                    RewriteCond %{REQUEST_URI} ^/index\.html$
                    RewriteCond %{DOCUMENT_ROOT}/index.html -f
                    RewriteRule ^(index\.html)?$ index.html [L]

                    # 栏目页和内容页转发到html目录
                    # 1. /xxx/ 或 /xxx => /html/xxx/index.html
                    RewriteCond %{REQUEST_FILENAME} !-f
                    RewriteCond %{REQUEST_FILENAME} !-d
                    RewriteCond %{DOCUMENT_ROOT}/html/\$1/index.html -f
                    RewriteRule ^([a-zA-Z0-9\-_]+)/?$ html/\$1/index.html [L]

                    # 2. /xxx/yyy.html => /html/xxx/yyy.html
                    RewriteCond %{REQUEST_FILENAME} !-f
                    RewriteCond %{REQUEST_FILENAME} !-d
                    RewriteCond %{DOCUMENT_ROOT}/html/\$1/\$2 -f
                    RewriteRule ^([a-zA-Z0-9\-_]+)/([a-zA-Z0-9\-_]+\.html)$ html/\$1/\$2 [L]

                    </IfModule>
                    HTA;

                    file_put_contents(root_path() . '.htaccess', $htaccess);
                }
            }

            $oldPath = Db::name('config')->where('id', 1)->value('admin_path');
            if($data['admin_path']!==$oldPath){
                // 路径已更改
                if ($data['admin_path'] != $oldPath) {
                    // 检查新路径合法性
                    if (empty($data['admin_path']) || !preg_match('/^[a-zA-Z0-9_]+$/', $data['admin_path'])) {
                        return json(['msg' => '后台登录路径不能为空，且只允许字母、数字和下划线', 'status' => 'fail']);
                    }
                    // 删除旧入口文件
                    if (file_exists(root_path() . $oldPath . '.php')) {
                        @unlink(root_path() . $oldPath . '.php');
                    }
                    // 创建新的入口文件
                    $this->createAdminEntry($data['admin_path']);
                }
                $config = Db::name('config')->where('id', 1)->strict(false)->save($data);
                if ($config) {
                    session(null);
                    return json(['msg' => '更新成功，操作修改了登录路径，请重新登录！', 'status' => 'success']);
                } else {
                    return json(['msg' => '更新失败', 'status' => 'fail']);
                }
            }

            $config=Db::name('config')->where('id',1)->strict(false)->save($data);
            if($config){
                // 日志记录（仅记录变更，不影响原有逻辑）
                if (!empty($logArr)) {
                    $this->logAction('更新配置参数:' . json_encode($logArr, JSON_UNESCAPED_UNICODE));
                }
                return json(['msg'=>'更新成功','status'=>'success']);
            }else{
                return json(['msg'=>'更新失败','status'=>'fail']);
            }
        }
        
        // 检查 $config['search'] 是否存在并且不为 null
        $search = isset($config['search']) ? explode(',', $config['search']) : [];
       
        $model=Db::name('model')->where('status',1)->where('type',2)->order('sort asc,id asc')->select();
        


        View::assign([
            'config'=>$config,
            'search'=>$search,
            'model'=>$model,
        ]);
        return View::fetch('index');
    }

    /**
     * 创建后台入口文件
     * @param string $path 入口文件名称（不含.php）
     * @return bool
     */
    private function createAdminEntry($path)
    {
            // 入口文件内容模板
            $entryTemplate = <<<'EOT'
        <?php
        // [ 应用入口文件 ]
        namespace think;

        require __DIR__ . '/vendor/autoload.php';

        // 执行HTTP应用并响应
        $http = (new App())->http;

        $response = $http->name('admin')->run();

        $response->send();

        $http->end($response);
        EOT;

            // 写入新入口文件
            $filePath = root_path() . $path . '.php';
            return file_put_contents($filePath, $entryTemplate) !== false;
    }

    /**
     * 发送测试邮件
     */
    public function send_test_email()
    {
        try {
            $data = input('post.');
            
            // 验证必要参数
            if (empty($data['smtp_server']) || empty($data['smtp_username']) || 
                empty($data['smtp_password']) || empty($data['smtp_username_test'])) {
                return json(['status' => 'fail', 'msg' => '邮件配置信息不完整！']);
            }

            // 邮件配置
            $smtpHost = $data['smtp_server'];
            $smtpPort = !empty($data['smtp_port']) ? intval($data['smtp_port']) : 587;
            $smtpUsername = $data['smtp_username'];
            $smtpPassword = $data['smtp_password'];
            $toEmail = $data['smtp_username_test'];
            
            // 邮件内容
            $subject = "CMS系统邮件配置测试 - " . date('Y-m-d H:i:s');
            $body = "这是一封测试邮件，用于验证您的邮件配置是否正确。\n\n";
            $body .= "发送时间：" . date('Y-m-d H:i:s') . "\n";
            $body .= "SMTP服务器：" . $smtpHost . "\n";
            $body .= "SMTP端口：" . $smtpPort . "\n";
            $body .= "发送邮箱：" . $smtpUsername . "\n";
            $body .= "接收邮箱：" . $toEmail . "\n\n";
            $body .= "如果您收到这封邮件，说明邮件配置正确！";
            
            // 发送邮件
            $this->sendSMTPMail($smtpHost, $smtpPort, $smtpUsername, $smtpPassword, $toEmail, $subject, $body);
            
            return json(['status' => 'success', 'msg' => '测试邮件发送成功，请检查收件箱！']);
            
        } catch (Exception $e) {
            return json(['status' => 'fail', 'msg' => '邮件发送失败：' . $e->getMessage()]);
        }
    }

    /**
     * SMTP邮件发送
     */
    private function sendSMTPMail($host, $port, $username, $password, $to, $subject, $body)
    {
        // 根据端口选择连接方式
        if ($port == 465) {
            // SSL连接
            $socket = fsockopen('ssl://' . $host, $port, $errno, $errstr, 30);
        } else {
            // 普通连接（587端口会后续升级为TLS）
            $socket = fsockopen($host, $port, $errno, $errstr, 30);
        }
        
        if (!$socket) {
            throw new Exception("无法连接到SMTP服务器: $errstr ($errno)");
        }
        
        // 读取服务器欢迎消息
        $this->getResponse($socket);
        
        // EHLO握手
        fwrite($socket, "EHLO localhost\r\n");
        $this->getResponse($socket);
        
        // 如果是587端口，启用STARTTLS
        if ($port == 587) {
            fwrite($socket, "STARTTLS\r\n");
            $response = $this->getResponse($socket);
            
            // 启用TLS加密
            $crypto_method = STREAM_CRYPTO_METHOD_TLS_CLIENT;
            if (defined('STREAM_CRYPTO_METHOD_TLSv1_2_CLIENT')) {
                $crypto_method = STREAM_CRYPTO_METHOD_TLSv1_2_CLIENT;
            }
            
            if (!stream_socket_enable_crypto($socket, true, $crypto_method)) {
                fclose($socket);
                throw new Exception("无法启用TLS加密连接");
            }
            
            // TLS握手后重新EHLO
            fwrite($socket, "EHLO localhost\r\n");
            $this->getResponse($socket);
        }
        
        // AUTH LOGIN认证
        fwrite($socket, "AUTH LOGIN\r\n");
        $this->getResponse($socket);
        
        // 发送用户名
        fwrite($socket, base64_encode($username) . "\r\n");
        $this->getResponse($socket);
        
        // 发送密码
        fwrite($socket, base64_encode($password) . "\r\n");
        $this->getResponse($socket);
        
        // 设置发送方
        fwrite($socket, "MAIL FROM: <$username>\r\n");
        $this->getResponse($socket);
        
        // 设置接收方
        fwrite($socket, "RCPT TO: <$to>\r\n");
        $this->getResponse($socket);
        
        // 开始邮件数据
        fwrite($socket, "DATA\r\n");
        $this->getResponse($socket);
        
        // 邮件头和内容
        $mail_data = "From: $username\r\n";
        $mail_data .= "To: $to\r\n";
        $mail_data .= "Subject: =?UTF-8?B?" . base64_encode($subject) . "?=\r\n";
        $mail_data .= "MIME-Version: 1.0\r\n";
        $mail_data .= "Content-Type: text/plain; charset=UTF-8\r\n";
        $mail_data .= "Content-Transfer-Encoding: 8bit\r\n\r\n";
        $mail_data .= $body . "\r\n";
        $mail_data .= ".\r\n";
        
        fwrite($socket, $mail_data);
        $this->getResponse($socket);
        
        // 退出
        fwrite($socket, "QUIT\r\n");
        $this->getResponse($socket);
        
        fclose($socket);
    }

    /**
     * 获取SMTP服务器响应
     */
    private function getResponse($socket)
    {
        $response = '';
        while (($line = fgets($socket, 515)) !== false) {
            $response .= $line;
            // SMTP响应以空格结尾表示结束，以-连接表示多行响应继续
            if (isset($line[3]) && $line[3] == ' ') {
                break;
            }
        }
        
        $code = substr($response, 0, 3);
        if ($code >= '400') {
            throw new Exception("SMTP错误 ($code): $response");
        }
        
        return $response;
    }


    

   
}
