<?php
declare(strict_types=1);

namespace app\home\controller;

use think\facade\Db;

class Sitemap
{
    private const DIR_NAME  = 'sitemap';
    private const FILE_NAME = 'sitemap.xml';
    private const ROOT_FILE = 'sitemap.xml';
    private const FLAG_FILE = 'changed.flag';
    private const LOCK_FILE = 'building.lock';
    private const INTERVAL  = 86400; // 1 天更新一次

    // 立即重建（供后台清缓存 / 业务调用）
    public function rebuildImmediately(): bool
    {
        $dir  = runtime_path() . self::DIR_NAME . DIRECTORY_SEPARATOR;
        if (!is_dir($dir) && !@mkdir($dir, 0777, true)) {
            $this->writeLog('目录创建失败: ' . $dir);
            return false;
        }
        $file = $dir . self::FILE_NAME;
        try {
            $urls = $this->gatherUrls();
            if (!$urls) {
                $this->writeLog('未收集到任何 URL，生成终止');
                return false;
            }
            $xml  = $this->buildXml($urls);
            $ok = @file_put_contents($file, $xml);
            if ($ok === false) {
                $this->writeLog('写入文件失败: ' . $file);
                return false;
            }
            // 额外写入根目录文件
            $this->writeRootSitemap($xml);
            return true;
        } catch (\Throwable $e) {
            $this->writeLog('异常: ' . $e->getMessage());
            return false;
        }
    }

    // 标记有变更（供栏目/内容控制器调用）
    public function markChanged(): void
    {
        $dir = runtime_path() . self::DIR_NAME . DIRECTORY_SEPARATOR;
        if (!is_dir($dir)) { @mkdir($dir, 0777, true); }

        $flag     = $dir . self::FLAG_FILE;
        $lockFile = $dir . self::LOCK_FILE;
        $rootFile = rtrim(root_path(), DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . self::ROOT_FILE;

        // 若根目录还没有 sitemap.xml，立即尝试生成（不等待1小时）
        if (!is_file($rootFile)) {
            // 简单锁，避免并发多次生成
            if (is_file($lockFile) && (time() - filemtime($lockFile)) < 300) {
                // 已有构建在 5 分钟窗口内，仍然写标记文件，后续访问可继续判断
                @touch($flag);
                return;
            }
            @touch($lockFile);
            try {
                if ($this->rebuildImmediately()) {
                    // 生成成功后无需标记（当前已是最新）
                    @unlink($flag);
                    return;
                }
            } catch (\Throwable $e) {
                // 忽略异常，继续走标记逻辑
            } finally {
                @unlink($lockFile);
            }
            // 若到这里说明生成失败，继续写标记，后续仍可按 lazy 逻辑重试
        }

        // 根目录已有文件 或 前面立即生成失败，则只做变更标记，交由 lazyGenerate() 在 >=1 小时后处理
        @touch($flag);
    }

    // 前台任意访问时可调用此方法实现“超过1小时自动重建”
    public function lazyGenerate(): void
    {
        $dir = runtime_path() . self::DIR_NAME . DIRECTORY_SEPARATOR;
        if (!is_dir($dir)) { return; }
        $flag = $dir . self::FLAG_FILE;
        if (!is_file($flag)) { return; } // 没有变更
        $lastChange = filemtime($flag) ?: 0;
        if ((time() - $lastChange) < self::INTERVAL) { return; } // 未超时

        $lock = $dir . self::LOCK_FILE;
        if (is_file($lock) && (time() - filemtime($lock)) < 300) { return; } // 5分钟内已有构建
        @touch($lock);
        try {
            if ($this->rebuildImmediately()) {
                @unlink($flag); // 成功后清除标记
            }
        } catch (\Throwable $e) {
            // 静默
        } finally {
            @unlink($lock);
        }
    }

    private function writeLog(string $msg): void
    {
        $dir  = runtime_path() . self::DIR_NAME . DIRECTORY_SEPARATOR;
        if (!is_dir($dir)) { return; }
        $line = '[' . date('Y-m-d H:i:s') . '] ' . $msg . "\n";
        @file_put_contents($dir . 'rebuild.log', $line, FILE_APPEND);
    }

    private function gatherUrls(): array
    {
        $domain = $this->getDomain();
        $urls   = [];
        $push = function(string $path, string $lastmod, string $priority) use (&$urls, $domain) {
            $path = '/' . ltrim($path, '/');
            $full = rtrim($domain, '/') . $path;
            $urls[$full] = [
                'loc' => htmlspecialchars($full, ENT_QUOTES, 'UTF-8'),
                'lastmod' => $lastmod,
                'priority' => $priority,
            ];
        };

        // 首页
        $push('/', date('Y-m-d'), '1.0');

        // ---- 动态时间字段探测配置 ----
        $catDateCandidates  = ['create_time','addtime','time','date','updatetime'];
        $contentDateCandidates = ['create_time','addtime','time','date','updatetime'];

        $catDateFields     = $this->detectExistingFields('category', $catDateCandidates); // 分类表实际存在的时间相关字段
        $contentDateFields = $this->detectExistingFields('content',  $contentDateCandidates); // 内容表实际存在的时间相关字段

        // 栏目（分类）获取
        $catSelectFields = 'id,model,link,status';
        if ($catDateFields) $catSelectFields .= ',' . implode(',', $catDateFields);
        $categories = Db::name('category')
            ->order('id asc')
            ->field($catSelectFields)
            ->select()->toArray();

        // 取模型 type，用于区分优先级（若表结构中有 model 字段）
        $modelMap = [];
        if ($categories) {
            $modelIds = array_unique(array_column($categories, 'model'));
            if ($modelIds) {
                $modelMap = Db::name('model')->whereIn('id', $modelIds)->column('type', 'id');
            }
        }

        foreach ($categories as $cat) {
            // 若有 status 且=0则忽略（旧表结构无 status 字段时继续）
            if (isset($cat['status']) && (int)$cat['status'] !== 1) continue;
            $link = $cat['link'] ?? '';
            if (!$link) continue;
            $raw = '';
            foreach ($catDateFields as $f) { // 依顺序取第一个非空
                if (!empty($cat[$f])) { $raw = $cat[$f]; break; }
            }
            $last = $this->formatDate($raw ?: time());
            $type = $modelMap[$cat['model']] ?? null;
            $push($link, $last, $type == 1 ? '0.8' : '0.7');
        }

        // 内容列表
        $contentSelectFields = 'id,link,status';
        if ($contentDateFields) $contentSelectFields .= ',' . implode(',', $contentDateFields);
        $contents = Db::name('content')
            ->order('id desc')
            ->field($contentSelectFields)
            ->select()->toArray();

        foreach ($contents as $row) {
            if (isset($row['status']) && (int)$row['status'] !== 1) continue; // 无 status 字段时不影响
            $link = $row['link'] ?? '';
            if (!$link) continue;
            $raw = '';
            foreach ($contentDateFields as $f) {
                if (!empty($row[$f])) { $raw = $row[$f]; break; }
            }
            $last = $this->formatDate($raw ?: time());
            $push($link, $last, '0.5');
        }

        return $urls;
    }

    /**
     * 探测表中实际存在的字段（按给定候选顺序返回交集）
     */
    private function detectExistingFields(string $table, array $candidates): array
    {
        try {
            $tableName = Db::name($table)->getTable();
            $columns = Db::query("SHOW COLUMNS FROM `{$tableName}`");
            if (!$columns) return [];
            $existing = array_column($columns, 'Field');
            return array_values(array_intersect($candidates, $existing));
        } catch (\Throwable $e) {
            return [];
        }
    }

    private function buildXml(array $urls): string
    {
        $xml = [];
        $xml[] = '<?xml version="1.0" encoding="UTF-8"?>';
        $xml[] = '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">';
        foreach ($urls as $u) {
            $xml[] = '  <url>';
            $xml[] = '    <loc>' . $u['loc'] . '</loc>';
            $xml[] = '    <lastmod>' . $u['lastmod'] . '</lastmod>';
            $xml[] = '    <changefreq>daily</changefreq>';
            $xml[] = '    <priority>' . $u['priority'] . '</priority>';
            $xml[] = '  </url>';
        }
        $xml[] = '</urlset>';
        return implode("\n", $xml);
    }

    private function getDomain(): string
    {
        $system = Db::name('system')->where('id', 1)->find();
        return isset($system['domain']) && $system['domain'] ? rtrim($system['domain'], '/') : rtrim(request()->domain(), '/');
    }

    private function formatDate($value): string
    {
        if (!$value) return date('Y-m-d');
        if (is_numeric($value)) return date('Y-m-d', (int)$value);
        $ts = strtotime((string)$value);
        return $ts ? date('Y-m-d', $ts) : date('Y-m-d');
    }

    // 写根目录 sitemap.xml
    private function writeRootSitemap(string $xml): void
    {
        try {
            $rootFile = rtrim(root_path(), DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . self::ROOT_FILE;
            $ok = @file_put_contents($rootFile, $xml);
            if ($ok === false) {
                $this->writeLog('根目录 sitemap 写入失败: ' . $rootFile);
            } else {
                $this->writeLog('根目录 sitemap 写入成功: ' . $rootFile);
            }
        } catch (\Throwable $e) {
            $this->writeLog('根目录 sitemap 写入异常: ' . $e->getMessage());
        }
    }
}
