<?php

namespace app\admin\controller;

use app\BaseController;
use think\facade\Db;
use think\File;

use think\facade\Config;
use think\facade\Env;
use think\Request;


class Upload extends Common
{

	public function get_images()
	{
		header('Content-Type: application/json');
		// 获取分页参数
		$page = isset($_GET['page']) ? intval($_GET['page']) : 1;
		$pageSize = isset($_GET['pageSize']) ? intval($_GET['pageSize']) : 10;
		$groupId = isset($_GET['groupId']) ? intval($_GET['groupId']) : null;
		
		// 获取排序参数，默认按批次排序
		$sortBy = isset($_GET['sortBy']) ? $_GET['sortBy'] : 'batch';

		// 获取时间范围参数
		$dateRange = isset($_GET['dateRange']) ? trim($_GET['dateRange']) : '';
		$startDate = $endDate = '';
		if ($dateRange && strpos($dateRange, ' - ') !== false) {
			list($startDate, $endDate) = explode(' - ', $dateRange);
			$startDate = trim($startDate);
			$endDate = trim($endDate);
		}

		$metadataFile = './uploads/metadata.json';
		$images = [];
		$baseUploadDir = 'uploads/image/';

		if (file_exists($metadataFile)) {
			$metadata = json_decode(file_get_contents($metadataFile), true);

			if (is_array($metadata)) {
				// 遍历所有日期目录
				$dirs = glob($baseUploadDir . '*', GLOB_ONLYDIR);

				// 构建所有图片的路径映射
				$filePathMap = [];
				foreach ($dirs as $dir) {
					$files = glob($dir . '/*');
					foreach ($files as $file) {
						$basename = basename($file);
						$filePathMap[$basename] = $dir . '/';
					}
				}

				// 构建图片数组
				foreach ($metadata as $filename => $info) {
					// 检查是否需要按分组筛选
					$itemGroupId = isset($info['group_id']) ? intval($info['group_id']) : 0;
					if ($groupId !== null && $itemGroupId != $groupId) {
						continue;
					}

					// 时间范围过滤
					if ($startDate && $endDate && isset($info['upload_time'])) {
						$imgDate = substr($info['upload_time'], 0, 10);
						if ($imgDate < $startDate || $imgDate > $endDate) {
							continue;
						}
					}

					// 查找文件实际路径
					$dir = isset($filePathMap[$filename]) ? $filePathMap[$filename] : null;
					if (!$dir) continue;

					$url = '/' . $dir . $filename;
					$name = isset($info['name']) ? $info['name'] : $filename;

					// 获取图片尺寸
					$width = 0;
					$height = 0;
					if (file_exists($dir . $filename)) {
						$size = getimagesize($dir . $filename);
						if ($size) {
							$width = $size[0];
							$height = $size[1];
						}
					}

					$images[] = [
						'url' => $url,
						'name' => $name,
						'width' => $width,
						'height' => $height,
						'group_id' => $itemGroupId,
						'upload_time' => isset($info['upload_time']) ? $info['upload_time'] : '',
						'batch_number' => isset($info['batch_number']) ? intval($info['batch_number']) : 0 // 确保转换为整数
					];
				}

				// 根据排序参数进行排序
				if ($sortBy === 'name') {
					// 纯按文件名自然排序
					usort($images, function($a, $b) {
						return strnatcmp($a['name'], $b['name']);
					});
				} else if ($sortBy === 'batch') {
					usort($images, function($a, $b) {
						$batchA = intval($a['batch_number']);
						$batchB = intval($b['batch_number']);
						
						// 先按批次号倒序
						if ($batchA != $batchB) {
							return $batchB - $batchA;
						}
						
						// 同批次内：按文件名排序（nature-1.jpg, nature-2.jpg, nature-10.jpg）
						$nameA = $a['name'];
						$nameB = $b['name'];
						
						// 使用自然排序，会正确处理 nature-1, nature-2, nature-10 的顺序
						return strnatcasecmp($nameA, $nameB);
					});
				}
			}
		}

		// 计算总图片数量
		$total = count($images);

		// 分页处理
		if ($pageSize < 9999) {
			$offset = ($page - 1) * $pageSize;
			$images = array_slice($images, $offset, $pageSize);
		}

		$simple = isset($_GET['simple']) ? intval($_GET['simple']) : 0;
		if ($simple) {
			// 只返回url数组
			$urls = array_column($images, 'url');
			echo json_encode([
				'success' => true,
				'data' => $urls,
				'total' => $total
			]);
			exit;
		}

		// 返回结果
		echo json_encode([
			'success' => true,
			'data' => $images,
			'total' => $total,
			'page' => $page,
			'pageSize' => $pageSize,
			'groupId' => $groupId
		]);
		exit;
	}

	/**
	 * 获取指定图片的排序结果（跨分组）
	 */
	public function get_all_images_for_sorting()
	{
		try {
			$selectedImages = input('selectedImages');
			$sortBy = input('sortBy', 'batch');

			if (!$selectedImages) {
				return json(['success' => false, 'message' => '缺少选中图片参数']);
			}

			$selectedImagesArray = json_decode($selectedImages, true);
			if (!is_array($selectedImagesArray) || empty($selectedImagesArray)) {
				return json(['success' => false, 'message' => '选中图片参数格式错误']);
			}

			// 构建查询条件：只获取选中的图片
			$whereConditions = [];
			foreach ($selectedImagesArray as $imagePath) {
				$whereConditions[] = ['url', '=', $imagePath];
			}

			if (empty($whereConditions)) {
				return json(['success' => true, 'data' => []]);
			}

			// 获取选中图片的详细信息
			$query = Db::table('upload_images')->whereOr($whereConditions);

			// 应用排序
			if ($sortBy === 'batch') {
				$query->order('batch_number DESC, name ASC');
			} else {
				$query->order('id DESC');
			}

			$images = $query->select()->toArray();

			// 补充可能缺失的字段
			foreach ($images as &$image) {
				if (!isset($image['width']) || !isset($image['height'])) {
					$imagePath = public_path() . ltrim($image['url'], '/');
					if (file_exists($imagePath)) {
						$imageInfo = getimagesize($imagePath);
						if ($imageInfo) {
							$image['width'] = $imageInfo[0];
							$image['height'] = $imageInfo[1];
						}
					}
				}
			}

			return json([
				'success' => true,
				'data' => $images,
				'total' => count($images)
			]);

		} catch (\Exception $e) {
			return json(['success' => false, 'message' => '获取图片排序失败: ' . $e->getMessage()]);
		}
	}

	public function get_image_groups()
	{
		// 分组数据JSON文件路径
		$dataFile = './uploads/image_groups.json';
		$imageDbFile = './uploads/images_db.json'; // 或其他存储图片的JSON文件

		// 确保目录存在
		if (!is_dir('./uploads')) {
			mkdir('./uploads', 0755, true);
		}

		
		if (!file_exists($dataFile)) {
			// 创建默认分组
			$data = [
				'groups' => [
					[
						'id' => 0,
						'name' => '默认分组',
						'count' => 0
					]
				]
			];
			file_put_contents($dataFile, json_encode($data, JSON_PRETTY_PRINT));
		} else {
			$data = json_decode(file_get_contents($dataFile), true);
		}
		
		// 重新计算每个分组的图片数量
		$this->recountGroupImages($data);
		
		// 保存更新后的数据
		file_put_contents($dataFile, json_encode($data, JSON_PRETTY_PRINT));
		
		header('Content-Type: application/json');
		echo json_encode($data);
		exit;
		
	}
	

	public function get_image_groups_save()
	{
		$dataFile = './uploads/image_groups.json';
		$imageDbFile = './images_db.json'; // 或其他存储图片的JSON文件

		// 确保目录存在
		if (!is_dir('./uploads')) {
			mkdir('./uploads', 0755, true);
		}

		$jsonData = file_get_contents('php://input');
		$data = json_decode($jsonData, true);
		
		if ($data) {
			// 判断是否有新增分组
			$oldGroups = [];
			if (file_exists($dataFile)) {
				$oldData = json_decode(file_get_contents($dataFile), true);
				if (isset($oldData['groups'])) {
					foreach ($oldData['groups'] as $g) {
						$oldGroups[$g['id']] = $g['name'];
					}
				}
			}
			$newGroups = [];
			if (isset($data['groups'])) {
				foreach ($data['groups'] as $g) {
					$newGroups[$g['id']] = $g['name'];
				}
			}
			// 新增分组日志
			foreach ($newGroups as $gid => $gname) {
				if (!isset($oldGroups[$gid])) {
					$this->logAction('新增分组: ' . $gname . ' (ID:' . $gid . ')');
				}
			}
			// 删除分组日志
			foreach ($oldGroups as $gid => $gname) {
				if (!isset($newGroups[$gid])) {
					$this->logAction('删除分组: ' . $gname . ' (ID:' . $gid . ')');
				}
			}
			// 修改分组日志（分组名变更）
			foreach ($newGroups as $gid => $gname) {
				if (isset($oldGroups[$gid]) && $oldGroups[$gid] !== $gname) {
					$this->logAction('修改分组: ' . $oldGroups[$gid] . ' 改为 ' . $gname . ' (ID:' . $gid . ')');
				}
			}
			// 重新计算每个分组的图片数量
			$this->recountGroupImages($data);
			// 保存更新后的数据
			file_put_contents($dataFile, json_encode($data, JSON_PRETTY_PRINT));
			header('Content-Type: application/json');
			echo json_encode(['success' => true]);
		} else {
			header('Content-Type: application/json');
			echo json_encode(['success' => false, 'message' => '数据解析失败']);
		}
		exit;
		
	}

	public function get_image_group()
	{
		$image = isset($_GET['image']) ? $_GET['image'] : '';
		if (!$image) {
			echo json_encode(['success' => false, 'message' => '缺少图片参数']);
			exit;
		}

		// 只取文件名部分
		$imageFile = basename($image);

		// 读取 metadata.json
		$metaFile = './uploads/metadata.json';
		if (!file_exists($metaFile)) {
			echo json_encode(['success' => false, 'message' => 'metadata.json不存在']);
			exit;
		}
		$meta = json_decode(file_get_contents($metaFile), true);
		if (!$meta || !isset($meta[$imageFile])) {
			echo json_encode(['success' => true, 'groupId' => 0]);
			exit;
		}
		echo json_encode(['success' => true, 'groupId' => intval($meta[$imageFile]['group_id'])]);
		exit;
	}

	//移动分组
	public function move_images()
	{
		$metadataFile = './uploads/metadata.json';

		// 检查必要参数
		if (!isset($_POST['imagePaths']) || !isset($_POST['targetGroupId'])) {
			echo json_encode([
				'success' => false,
				'message' => '缺少必要参数'
			]);
			exit;
		}

		// 获取参数
		$imagePaths = json_decode($_POST['imagePaths'], true);
		$targetGroupId = intval($_POST['targetGroupId']);

		// 验证图片路径数组
		if (!is_array($imagePaths) || empty($imagePaths)) {
			echo json_encode([
				'success' => false,
				'message' => '无效的图片路径'
			]);
			exit;
		}

		// 检查元数据文件
		if (!file_exists($metadataFile)) {
			echo json_encode([
				'success' => false,
				'message' => '图片元数据文件不存在'
			]);
			exit;
		}

		// 读取元数据
		$metadata = json_decode(file_get_contents($metadataFile), true);
		if (!is_array($metadata)) {
			echo json_encode([
				'success' => false,
				'message' => '无法解析元数据'
			]);
			exit;
		}

		// 转换图片路径为文件名
		$movedCount = 0;
		$movedNames = [];
		foreach ($imagePaths as $path) {
			$filename = basename($path);
			// 跳过不存在于元数据中的文件
			if (!isset($metadata[$filename])) {
				$strippedPath = str_replace('./uploads/', '', $path);
				if (isset($metadata[$strippedPath])) {
					$filename = $strippedPath;
				} else {
					continue;
				}
			}
			// 更新分组ID
			$metadata[$filename]['group_id'] = $targetGroupId;
			$movedCount++;
			$movedNames[] = isset($metadata[$filename]['name']) ? $metadata[$filename]['name'] : $filename;
		}
		// 保存更新后的元数据
		if ($movedCount > 0) {
			file_put_contents($metadataFile, json_encode($metadata, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT));
			// 更新分组计数
			$this->updateAllGroupCounts();
			// 日志
			$this->logAction('移动分组: 目标分组ID:' . $targetGroupId . '，图片名: ' . implode('，', $movedNames));
			echo json_encode([
				'success' => true,
				'message' => "成功移动 $movedCount 张图片到指定分组"
			]);
		} else {
			echo json_encode([
				'success' => false,
				'message' => '未能移动任何图片'
			]);
		}
	}

	//删除分组，分组的图片移动到默认分组
	public function move_group_images()
	{
		// 获取POST参数
		$sourceGroupId = isset($_POST['sourceGroupId']) ? intval($_POST['sourceGroupId']) : 0;
		$targetGroupId = isset($_POST['targetGroupId']) ? intval($_POST['targetGroupId']) : 0;



		if ($sourceGroupId === $targetGroupId) {
			echo json_encode(['success' => false, 'message' => '源分组和目标分组不能相同']);
			exit;
		}

		// 查找正确的图片数据库文件
		$possibleFiles = [
			'./uploads/metadata.json',
			'./images_db.json',
			'./data/images.json',
			'./uploadimages.json',
			'./images.json',
			'./data/uploadimages.json',
			'./data/metadata.json'
		];

		$imagesDbFile = null;
		foreach ($possibleFiles as $file) {
			if (file_exists($file)) {
				$imagesDbFile = $file;
				break;
			}
		}

		if (!$imagesDbFile) {

			// 尝试查找任何可能的JSON文件作为备选
			$jsonFiles = glob('./*.json');
			$dataJsonFiles = glob('./data/*.json');
			$allJsonFiles = array_merge($jsonFiles, $dataJsonFiles);
			
			// 如果有任何JSON文件，尝试第一个
			if (!empty($allJsonFiles)) {
				$imagesDbFile = $allJsonFiles[0];
			} else {
				echo json_encode(['success' => false, 'message' => '无法找到图片数据库文件']);
				exit;
			}
		}

		// 读取图片数据库文件
		$imagesDbContent = file_get_contents($imagesDbFile);

		// 解析JSON
		$imagesDb = json_decode($imagesDbContent, true);
		if (!$imagesDb) {
			echo json_encode(['success' => false, 'message' => '图片数据库解析失败']);
			exit;
		}


		// 尝试找出图片数组的正确位置
		$imagesArray = null;
		if (isset($imagesDb['images'])) {
			$imagesArray = &$imagesDb['images'];
		} else {
			// 如果顶层就是图片数组
			$imagesArray = &$imagesDb;
		}

		// 更新分组ID，使用非严格比较(==)
		$changed = false;
		$movedCount = 0;

		// 检查是使用'groupId'还是'group_id'作为分组ID字段
		$groupIdField = null;
		$sampleImage = reset($imagesArray);

		// 尝试不同的可能字段名称
		$possibleFields = ['groupId', 'group_id', 'groupid', 'group-id', 'group'];
		foreach ($possibleFields as $field) {
			if ($sampleImage && isset($sampleImage[$field])) {
				$groupIdField = $field;
				break;
			}
		}

		// 如果没有找到已知的字段，尝试记录所有可用字段
		if (!$groupIdField && $sampleImage) {;
			// 尝试猜测可能的字段
			foreach (array_keys($sampleImage) as $field) {
				if (stripos($field, 'group') !== false || stripos($field, 'category') !== false) {
					$groupIdField = $field;
					break;
				}
			}
		}

		// 遍历所有图片
		foreach ($imagesArray as &$image) {
			// 如果找到了有效的分组字段名
			if ($groupIdField && isset($image[$groupIdField]) && $image[$groupIdField] == $sourceGroupId) {
				$oldGroupId = $image[$groupIdField];
				$image[$groupIdField] = $targetGroupId;
				$changed = true;
				$movedCount++;
			}
		}

		// 保存修改后的数据库
		if ($changed) {
			$newContent = json_encode($imagesDb, JSON_PRETTY_PRINT);
			$result = file_put_contents($imagesDbFile, $newContent);
			
			if ($result === false) {
				echo json_encode(['success' => false, 'message' => '保存图片数据库失败']);
				exit;
			}
			
		}

		// 尝试查找分组数据文件
		$possibleGroupFiles = [
			'./uploads/image_groups.json',
			'./image_groups.json',
			'./groups.json',
			'./data/groups.json'
		];

		$groupsFile = null;
		foreach ($possibleGroupFiles as $file) {
			if (file_exists($file)) {
				$groupsFile = $file;
				break;
			}
		}

		if (!$groupsFile) {
			// 默认使用第一个路径
			$groupsFile = $possibleGroupFiles[0];
		}

		// 从分组数据中删除源分组
		if (file_exists($groupsFile)) {
			$groups = json_decode(file_get_contents($groupsFile), true);
			if ($groups && isset($groups['groups'])) {
				// 在删除前记录分组
				
				// 查找源分组和目标分组
				$found = false;
				$targetFound = false;
				$targetIndex = -1;
				
				foreach ($groups['groups'] as $key => $group) {
					if ($group['id'] == $sourceGroupId) {
						// 日志：分组被物理删除
						$this->logAction('删除分组: ' . (isset($group['name']) ? $group['name'] : '') . ' (ID:' . $group['id'] . ')');
						unset($groups['groups'][$key]);
						$found = true;
					}
					if ($group['id'] == $targetGroupId) {
						$targetFound = true;
						$targetIndex = $key;
					}
				}
				
				
				// 重建索引
				$groups['groups'] = array_values($groups['groups']);
				
				// 更新目标分组的计数
				if ($targetFound && $movedCount > 0) {
					$oldCount = $groups['groups'][$targetIndex]['count'];
					$groups['groups'][$targetIndex]['count'] = $oldCount + $movedCount;
				}
				
				// 保存回文件
				$result = file_put_contents($groupsFile, json_encode($groups, JSON_PRETTY_PRINT));
			}
		} else {
			
			// 创建基本的分组结构
			$groups = [
				'groups' => [
					[
						'id' => 0,
						'name' => '默认分组',
						'count' => $movedCount
					]
				]
			];
			
			// 确保data目录存在
			$dir = dirname($groupsFile);
			if (!is_dir($dir)) {
				mkdir($dir, 0755, true);
			}
			
			// 保存新创建的分组文件
			$result = file_put_contents($groupsFile, json_encode($groups, JSON_PRETTY_PRINT));
		}

		// 更新所有分组的计数 - 重新扫描所有图片，确保计数准确
		$this->updateAllGroupCounts($imagesDbFile, $groupsFile, $groupIdField);

		echo json_encode([
			'success' => true, 
			'movedCount' => $movedCount,
			'message' => "成功将 $movedCount 张图片移动到默认分组"
		]);
	}
	

	public function get_directory_images()
	{
		$page = isset($_GET['page']) ? intval($_GET['page']) : 1;
		$pageSize = isset($_GET['pageSize']) ? intval($_GET['pageSize']) : 10;
		$directory = isset($_GET['directory']) ? $_GET['directory'] : 'uploads';

		// 目录安全性检查，确保只能访问uploads目录下的内容
		$baseDir = realpath('./uploads');
		$requestedDir = realpath('./' . $directory);

		// 检查请求的目录是否存在且在允许的范围内
		if (!$requestedDir || strpos($requestedDir, $baseDir) !== 0) {
			echo json_encode([
				'success' => false,
				'message' => '无效的目录路径',
				'data' => [],
				'total' => 0
			]);
			exit;
		}

		// 支持的图片类型
		$supportedExtensions = ['jpg', 'jpeg', 'png', 'gif', 'bmp', 'webp', 'svg'];

		// 获取指定目录中的所有图片
		$images = [];
		$files = scandir($requestedDir);

		foreach ($files as $file) {
			// 跳过.和..目录以及隐藏文件
			if ($file == '.' || $file == '..' || substr($file, 0, 1) === '.') {
				continue;
			}
			
			$filePath = $requestedDir . DIRECTORY_SEPARATOR . $file;
			
			// 只处理文件且是图片类型的文件
			if (is_file($filePath)) {
				$extension = strtolower(pathinfo($file, PATHINFO_EXTENSION));
				
				if (in_array($extension, $supportedExtensions)) {
					// 获取图片大小
					list($width, $height) = @getimagesize($filePath);
					
					// 相对路径
					$relativePath = str_replace('\\', '/', substr($filePath, strlen(realpath('.')) + 1));
					if (substr($relativePath, 0, 1) !== '/') {
						$relativePath = '/' . $relativePath;
					}
					
					$images[] = [
						'url' => $relativePath,
						'name' => $file,
						'width' => $width ?: 0,
						'height' => $height ?: 0,
						'size' => filesize($filePath),
						'lastModified' => filemtime($filePath)
					];
				}
			}
		}

		// 按最后修改时间倒序排序
		usort($images, function($a, $b) {
			return $b['lastModified'] - $a['lastModified'];
		});

		// 计算总数
		$total = count($images);

		// 分页
		$offset = ($page - 1) * $pageSize;
		$pageImages = array_slice($images, $offset, $pageSize);

		// 返回JSON格式的结果
		echo json_encode([
			'success' => true,
			'data' => $pageImages,
			'total' => $total
		]);
	}

	public function get_directory_structure()
	{
		// 获取请求的目录路径
		$basePath = isset($_GET['path']) ? $_GET['path'] : 'uploads';
		$basePath = realpath($basePath); // 转换为绝对路径

		// 安全检查：确保目录在允许范围内
		$allowedRoot = realpath('./uploads');
		if (!$basePath || strpos($basePath, $allowedRoot) !== 0) {
			echo json_encode([
				'success' => false,
				'message' => '无效的目录路径'
			]);
			exit;
		}

		// 递归获取目录结构，转为Layui Tree格式
		function scanDirectory($dir, $relativePath = '') {
			$result = [];
			
			if (!is_dir($dir)) {
				return $result;
			}
			
			$items = scandir($dir);
			foreach ($items as $item) {
				// 跳过.和..以及隐藏文件
				if ($item == '.' || $item == '..' || substr($item, 0, 1) === '.') {
					continue;
				}
				
				$path = $dir . DIRECTORY_SEPARATOR . $item;
				$currentPath = $relativePath ? $relativePath . '/' . $item : $item;
				
				if (is_dir($path)) {
					// 检查是否是最后一级目录（没有子目录的目录）
					$hasSubDirs = false;
					$subItems = scandir($path);
					foreach ($subItems as $subItem) {
						// 修正判断条件，确保正确检测子目录
						if ($subItem != '.' && $subItem != '..' && substr($subItem, 0, 1) !== '.' && is_dir($path . DIRECTORY_SEPARATOR . $subItem)) {
							$hasSubDirs = true;
							break;
						}
					}

					// 计算图片数
					$imgCount = countImagesInDir($path);

					// 创建目录节点
					$node = [
						'title' => $item . ($hasSubDirs ? '' : ' ('.$imgCount.')'),  // 只有最后一级目录显示图片数量
						'path' => $currentPath,
						'isLeaf' => false,  // 所有目录节点都不是叶子节点
						'spread' => false,  // 默认不展开
						'imgCount' => $imgCount // 保存图片数量
					];
					
					// 递归获取子目录
					$children = scanDirectory($path, $currentPath);
					if (!empty($children)) {
						$node['children'] = $children;
					}
					
					$result[] = $node;
				}
			}
			
			return $result;
		}

		// 计算目录中的图片文件数量
		function countImagesInDir($dir) {
			$count = 0;
			$imgExtensions = ['jpg', 'jpeg', 'png', 'gif', 'bmp', 'webp', 'svg'];
			
			if (is_dir($dir)) {
				$files = scandir($dir);
				foreach ($files as $file) {
					if ($file == '.' || $file == '..') continue;
					
					$filePath = $dir . DIRECTORY_SEPARATOR . $file;
					if (is_file($filePath)) {
						$ext = strtolower(pathinfo($file, PATHINFO_EXTENSION));
						if (in_array($ext, $imgExtensions)) {
							$count++;
						}
					}
				}
			}
			
			return $count;
		}

		try {
			// 创建根节点
			$treeData = [
				[
					'title' => basename($basePath),
					'path' => 'uploads',
					'isLeaf' => false,
					'spread' => true,
					'icon' => 'layui-icon layui-icon-folder',  // 根目录也使用文件夹图标
					'children' => scanDirectory($basePath, 'uploads')
				]
			];
			
			echo json_encode([
				'success' => true,
				'data' => $treeData
			]);
		} catch (Exception $e) {
			echo json_encode([
				'success' => false,
				'message' => $e->getMessage()
			]);
		}
		
	}

	//上传图片
	public function upload_images()
	{
		header('Content-Type: application/json');

		// 自动生成上传目录（按日期）
		$today = date('Ymd');
		$baseDir = "uploads/image";
		$uploadDir = "{$baseDir}/{$today}/";
		$metadataFile = './uploads/metadata.json';

		// 确保目录存在
		if (!is_dir('./uploads')) {
			if (!mkdir('./uploads', 0777, true)) {
				exit(json_encode(['success' => false, 'msg' => "创建uploads目录失败"]));
			}
		}

		// 关键修改：更安全的元数据文件初始化
		if (!file_exists($metadataFile)) {
			if (file_put_contents($metadataFile, json_encode([], JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT)) === false) {
				exit(json_encode(['success' => false, 'msg' => "创建元数据文件失败: $metadataFile"]));
			}
		} else {
			// 检查文件是否可读且内容有效
			$testContent = @file_get_contents($metadataFile);
			if ($testContent === false) {
				exit(json_encode(['success' => false, 'msg' => "无法读取元数据文件: $metadataFile"]));
			}
			
			// 验证JSON格式
			$testData = @json_decode($testContent, true);
			if ($testData === null && json_last_error() !== JSON_ERROR_NONE) {
				// JSON格式损坏，尝试备份并重新初始化
				$backupFile = $metadataFile . '.backup.' . date('YmdHis');
				@copy($metadataFile, $backupFile);
				
				if (file_put_contents($metadataFile, json_encode([], JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT)) === false) {
					exit(json_encode(['success' => false, 'msg' => "重新初始化元数据文件失败"]));
				}
			}
		}

		if (!is_dir($baseDir)) {
			if (!mkdir($baseDir, 0777, true)) {
				exit(json_encode(['success' => false, 'msg' => "创建基础目录失败: $baseDir"]));
			}
		}
		if (!is_dir($uploadDir)) {
			if (!mkdir($uploadDir, 0777, true)) {
				exit(json_encode(['success' => false, 'msg' => "创建上传目录失败: $uploadDir"]));
			}
		}

		// 获取分组ID，默认为0（默认分组）
		$groupId = isset($_POST['groupId']) ? intval($_POST['groupId']) : 0;

		// 关键修改：从前端接收批次号，如果没有则生成一个
		$currentBatchNumber = isset($_POST['batchNumber']) ? intval($_POST['batchNumber']) : time();
		
		$uploadedFiles = [];
		$errorFiles = [];
		$newMetadata = [];

		// 处理上传的文件
		if (isset($_FILES['file']['name'])) {
			if (is_array($_FILES['file']['name'])) {
				// 批量上传处理
				foreach ($_FILES['file']['name'] as $key => $name) {
					$tmpName = $_FILES['file']['tmp_name'][$key];
					$error = $_FILES['file']['error'][$key];
					$size = $_FILES['file']['size'][$key];

					if ($error === UPLOAD_ERR_OK) {
						$extension = strtolower(pathinfo($name, PATHINFO_EXTENSION));
						$timestamp = time();
						$rand = mt_rand(100000, 999999);
						$uniqueName = $timestamp . $rand . '.' . $extension;
						$filePath = $uploadDir . $uniqueName;

						if (move_uploaded_file($tmpName, $filePath)) {
							// 获取微秒级时间
							$microtime = microtime(true);
							$precise_time = date('Y-m-d H:i:s', floor($microtime)) . '.' . sprintf('%03d', round(($microtime - floor($microtime)) * 1000));
							
							// 使用前端传递的批次号
							$newMetadata[$uniqueName] = [
								'name' => $name,
								'upload_time' => $precise_time,
								'group_id' => $groupId,
								'batch_number' => $currentBatchNumber // 使用前端传递的批次号
							];
							
							$uploadedFiles[] = [
								'url' => '/'.$filePath,
								'name' => $name,
								'group_id' => $groupId,
								'batch_number' => $currentBatchNumber
							];
						} else {
							$errorMsg = "移动文件失败: " . (error_get_last()['message'] ?? '未知错误');
							$errorFiles[] = ['name' => $name, 'error' => $errorMsg];
						}
					} else {
						$errorMsg = "上传错误码: $error";
						$errorFiles[] = ['name' => $name, 'error' => $errorMsg];
					}
				}
			} else {
				// 单文件上传处理
				$name = $_FILES['file']['name'];
				$tmpName = $_FILES['file']['tmp_name'];
				$error = $_FILES['file']['error'];
				$size = $_FILES['file']['size'];

				if ($error === UPLOAD_ERR_OK) {
					$extension = strtolower(pathinfo($name, PATHINFO_EXTENSION));
					$timestamp = time();
					$rand = mt_rand(100000, 999999);
					$uniqueName = $timestamp . $rand . '.' . $extension;
					$filePath = $uploadDir . $uniqueName;

					if (move_uploaded_file($tmpName, $filePath)) {
						$microtime = microtime(true);
						$precise_time = date('Y-m-d H:i:s', floor($microtime)) . '.' . sprintf('%03d', round(($microtime - floor($microtime)) * 1000));
						
						// 使用前端传递的批次号
						$newMetadata[$uniqueName] = [
							'name' => $name,
							'upload_time' => $precise_time,
							'group_id' => $groupId,
							'batch_number' => $currentBatchNumber // 使用前端传递的批次号
						];
						
						$uploadedFiles[] = [
							'url' => '/'.$filePath,
							'name' => $name,
							'group_id' => $groupId,
							'batch_number' => $currentBatchNumber
						];
					} else {
						$errorMsg = "移动文件失败: " . (error_get_last()['message'] ?? '未知错误');
						$errorFiles[] = ['name' => $name, 'error' => $errorMsg];
					}
				} else {
					$errorMsg = "上传错误码: $error";
					$errorFiles[] = ['name' => $name, 'error' => $errorMsg];
				}
			}

			// 关键修改：超安全的元数据写入逻辑
			if (!empty($newMetadata)) {
				$maxRetries = 3;
				$retryCount = 0;
				$writeSuccess = false;

				while ($retryCount < $maxRetries && !$writeSuccess) {
					$retryCount++;
					
					// 创建备份文件名
					$backupFile = $metadataFile . '.backup.' . date('YmdHis') . '.' . $retryCount;
					
					try {
						// 第一步：创建当前文件的备份
						if (file_exists($metadataFile)) {
							if (!copy($metadataFile, $backupFile)) {
								if ($retryCount >= $maxRetries) {
									error_log("无法创建metadata备份文件: $backupFile");
								}
								usleep(100000); // 等待100ms后重试
								continue;
							}
						}

						// 第二步：安全读取现有数据
						$fp = fopen($metadataFile, 'c+');
						if (!$fp) {
							if ($retryCount >= $maxRetries) {
								error_log("无法打开metadata文件进行写入");
							}
							usleep(100000);
							continue;
						}

						// 第三步：获取独占锁
						$lockStart = microtime(true);
						$lockTimeout = 5; // 5秒超时
						
						while (!flock($fp, LOCK_EX | LOCK_NB)) {
							if (microtime(true) - $lockStart > $lockTimeout) {
								fclose($fp);
								if ($retryCount >= $maxRetries) {
									error_log("获取metadata文件锁超时");
								}
								usleep(100000);
								continue 2; // 继续外层循环
							}
							usleep(10000); // 等待10ms
						}

						// 第四步：读取现有数据
						rewind($fp);
						$fileSize = fstat($fp)['size'];
						$oldContent = '';
						if ($fileSize > 0) {
							$oldContent = fread($fp, $fileSize);
						}

						$oldMeta = [];
						if ($oldContent && trim($oldContent) !== '') {
							$oldMeta = json_decode($oldContent, true);
							if (!is_array($oldMeta)) {
								// JSON损坏，使用备份或初始化为空数组
								error_log("检测到metadata JSON损坏，使用备份数据");
								$oldMeta = [];
								
								// 尝试从最近的备份恢复
								$backupFiles = glob($metadataFile . '.backup.*');
								if (!empty($backupFiles)) {
									rsort($backupFiles); // 按时间倒序
									foreach ($backupFiles as $backup) {
										$backupContent = @file_get_contents($backup);
										if ($backupContent) {
											$backupData = @json_decode($backupContent, true);
											if (is_array($backupData)) {
												$oldMeta = $backupData;
												error_log("从备份文件恢复数据: $backup");
												break;
											}
										}
									}
								}
							}
						}
						
						// 第五步：合并新旧数据
						$allMetadata = array_merge($oldMeta, $newMetadata);
						
						// 第六步：生成新的JSON内容
						$newContent = json_encode($allMetadata, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
						if ($newContent === false) {
							flock($fp, LOCK_UN);
							fclose($fp);
							if ($retryCount >= $maxRetries) {
								error_log("JSON编码失败");
							}
							usleep(100000);
							continue;
						}

						// 第七步：写入新内容
						ftruncate($fp, 0);
						rewind($fp);
						$bytesWritten = fwrite($fp, $newContent);
						fflush($fp);
						
						// 第八步：验证写入
						if ($bytesWritten === strlen($newContent)) {
							$writeSuccess = true;
							
							// 验证写入的数据完整性
							rewind($fp);
							$verifyContent = fread($fp, strlen($newContent));
							$verifyData = @json_decode($verifyContent, true);
							
							if (!is_array($verifyData) || count($verifyData) !== count($allMetadata)) {
								error_log("写入验证失败，数据可能损坏");
								$writeSuccess = false;
							}
						}
						
						// 第九步：释放锁和文件句柄
						flock($fp, LOCK_UN);
						fclose($fp);
						
						if ($writeSuccess) {
							// 写入成功，清理临时备份
							@unlink($backupFile);
							
							// 清理过期的备份文件（保留最近5个）
							$allBackups = glob($metadataFile . '.backup.*');
							if (count($allBackups) > 5) {
								rsort($allBackups);
								$toDelete = array_slice($allBackups, 5);
								foreach ($toDelete as $old) {
									@unlink($old);
								}
							}
						} else {
							// 写入失败，尝试从备份恢复
							if (file_exists($backupFile)) {
								copy($backupFile, $metadataFile);
								@unlink($backupFile);
							}
						}

					} catch (Exception $e) {
						error_log("元数据写入异常: " . $e->getMessage());
						
						// 异常情况下恢复备份
						if (isset($fp) && is_resource($fp)) {
							@flock($fp, LOCK_UN);
							@fclose($fp);
						}
						
						if (file_exists($backupFile)) {
							copy($backupFile, $metadataFile);
							@unlink($backupFile);
						}
						
						if ($retryCount >= $maxRetries) {
							error_log("重试次数已达上限，元数据写入最终失败");
						}
					}
					
					if (!$writeSuccess) {
						usleep(200000); // 等待200ms后重试
					}
				}

				// 如果所有重试都失败了，记录错误但不影响上传结果的返回
				if (!$writeSuccess) {
					error_log("元数据写入最终失败，但文件上传成功");
					// 可以考虑发送邮件或其他通知方式
				}
			}
		}

		if (!empty($uploadedFiles)) {
			// 关键修改：对本次上传的文件按原始文件名进行自然排序
			usort($uploadedFiles, function($a, $b) {
				// 使用自然排序，正确处理 nature-1, nature-2, nature-10 的顺序
				return strnatcasecmp($a['name'], $b['name']);
			});
			// 调试输出 - 记录到日志
			$fileNames = array_map(function($f) { return $f['name']; }, $uploadedFiles);
			error_log('PHP上传文件排序后的顺序: ' . json_encode($fileNames));
			// 日志记录
			$this->logAction('上传图片，上传' . count($fileNames) . '张，图片名: ' . implode('，', $fileNames));
		}

		// 返回结果
		$response = [
			'success' => !empty($uploadedFiles),
			'files' => $uploadedFiles
		];

		if (!empty($errorFiles)) {
			$response['errors'] = $errorFiles;
		}

		if (!empty($uploadedFiles)) {
			$response['lastUploaded'] = reset($uploadedFiles);
		}

		echo json_encode($response);
	}

	/**
	 * 修复损坏的元数据文件
	 */
	public function repair_metadata()
	{
		header('Content-Type: application/json');
		
		$metadataFile = './uploads/metadata.json';
		$baseUploadDir = 'uploads/image/';
		
		try {
			// 检查是否需要修复
			$needRepair = false;
			$currentData = [];
			
			if (!file_exists($metadataFile)) {
				$needRepair = true;
			} else {
				$content = file_get_contents($metadataFile);
				$currentData = @json_decode($content, true);
				if (!is_array($currentData)) {
					$needRepair = true;
					$currentData = [];
				}
			}
			
			if ($needRepair) {
				// 扫描文件系统重建元数据
				$rebuiltData = [];
				
				// 遍历所有日期目录
				$dirs = glob($baseUploadDir . '*', GLOB_ONLYDIR);
				foreach ($dirs as $dir) {
					$files = glob($dir . '/*');
					foreach ($files as $file) {
						$filename = basename($file);
						$filemtime = filemtime($file);
						
						$rebuiltData[$filename] = [
							'name' => $filename,
							'upload_time' => date('Y-m-d H:i:s', $filemtime),
							'group_id' => 0,
							'batch_number' => $filemtime
						];
					}
				}
				
				// 合并现有数据和重建数据
				$finalData = array_merge($rebuiltData, $currentData);
				
				// 创建备份
				if (file_exists($metadataFile)) {
					copy($metadataFile, $metadataFile . '.backup.' . date('YmdHis'));
				}
				
				// 写入修复后的数据
				file_put_contents($metadataFile, json_encode($finalData, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT));
				
				echo json_encode([
					'success' => true,
					'message' => '元数据文件修复完成',
					'repaired_count' => count($rebuiltData),
					'total_count' => count($finalData)
				]);
			} else {
				echo json_encode([
					'success' => true,
					'message' => '元数据文件正常，无需修复',
					'total_count' => count($currentData)
				]);
			}
			
		} catch (Exception $e) {
			echo json_encode([
				'success' => false,
				'message' => '修复失败: ' . $e->getMessage()
			]);
		}
	}
	

	//删除图片
	public function delete_image()
	{
		header('Content-Type: application/json');

		$imagesDir = './uploads/'; // 图片目录
		$metadataFile = './uploads/metadata.json'; // 存储图片元数据的文件
		$groupsFile = './uploads/image_groups.json'; // 分组数据文件

		// 支持批量删除
		if (isset($_POST['imagePaths'])) {
			$imagePaths = json_decode($_POST['imagePaths'], true);
			$success = true;
			$messages = [];
			$deletedNames = [];
			foreach ($imagePaths as $imagePath) {
				if (strpos($imagePath, '/uploads/') === 0) {
					$imagePath = '.' . $imagePath;
				}
				if (!file_exists($imagePath)) {
					$success = false;
					$messages[] = "$imagePath 不存在";
					continue;
				}
				if (unlink($imagePath)) {
					$deletedNames[] = basename($imagePath);
					// 更新元数据
					if (file_exists($metadataFile)) {
						$metadata = json_decode(file_get_contents($metadataFile), true);
						$fileName = basename($imagePath);
						if (isset($metadata[$fileName])) {
							unset($metadata[$fileName]);
							file_put_contents($metadataFile, json_encode($metadata, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT));
						}
					}
				} else {
					$success = false;
					$messages[] = "$imagePath 删除失败";
				}
			}
			// 批量删除后更新所有分组的计数
			if ($success) {
				$this->updateAllGroupCounts();
				if (!empty($deletedNames)) {
					$this->logAction('批量删除图片，删除' . count($deletedNames) . '张，图片名: ' . implode('，', $deletedNames));
				}
				echo json_encode(['status' => '200', 'success' => true, 'message' => '图片已删除']);
			} else {
				echo json_encode(['status' => '400', 'success' => false, 'message' => implode('; ', $messages)]);
			}
			exit;
		}

		// 单张删除
		if (!isset($_POST['imagePath'])) {
			echo json_encode(['status' => '400', 'success' => false, 'message' => '未提供图片路径']);
			exit;
		}

		$imagePath = $_POST['imagePath'];
		if (strpos($imagePath, '/uploads/') === 0) {
			$imagePath = '.' . $imagePath;
		}

		if (!file_exists($imagePath)) {
			echo json_encode(['status' => '400', 'success' => false, 'message' => '图片文件不存在']);
			exit;
		}

		if (unlink($imagePath)) {
			$deletedName = basename($imagePath);
			if (file_exists($metadataFile)) {
				$metadata = json_decode(file_get_contents($metadataFile), true);
				$fileName = basename($imagePath);
				if (isset($metadata[$fileName])) {
					unset($metadata[$fileName]);
					file_put_contents($metadataFile, json_encode($metadata, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT));
				}
			}
			// 单张图片删除后更新所有分组的计数
			$this->updateAllGroupCounts();
			$this->logAction('删除图片，删除1张，图片名: ' . $deletedName);
			echo json_encode(['status' => '200', 'success' => true, 'message' => '图片已删除']);
		} else {
			echo json_encode(['status' => '400', 'success' => false, 'message' => '无法删除图片文件']);
		}

	}

	//上传文件
	public function upload_file()
	{
		header('Content-Type: application/json');
		// 自动生成上传目录（按日期）
		$today = date('Ymd');
		$baseDir = "uploads/file";
		$uploadDir = "{$baseDir}/{$today}/";

		if (!is_dir($baseDir)) mkdir($baseDir, 0777, true);
		if (!is_dir($uploadDir)) mkdir($uploadDir, 0777, true);

		if (isset($_FILES['file']['name'])) {
			$name = $_FILES['file']['name'];
			$tmpName = $_FILES['file']['tmp_name'];
			$error = $_FILES['file']['error'];

			if ($error === UPLOAD_ERR_OK) {
				$extension = strtolower(pathinfo($name, PATHINFO_EXTENSION));
				$uniqueName = time() . mt_rand(100000, 999999) . '.' . $extension;
				$filePath = $uploadDir . $uniqueName;

				if (move_uploaded_file($tmpName, $filePath)) {
					$this->logAction('上传文件，文件名: ' . $uniqueName);
					return json(['path'=>'/' . $filePath]);
				} else {
					echo json_encode(['success' => false, 'message' => '移动文件失败']);
					return;
				}
			} else {
				echo json_encode(['success' => false, 'message' => '上传错误码: ' . $error]);
				return;
			}
		} else {
			echo json_encode(['success' => false, 'message' => '未检测到上传文件']);
			return;
		}
		

		
	}

	//远程图片下载
	public function fetch_image()
	{
		header('Content-Type: application/json');

		$url = isset($_POST['url']) ? trim($_POST['url']) : '';
		if (!$url) {
			echo json_encode(['success' => false, 'message' => '图片地址不能为空']);
			exit;
		}

		// 校验图片链接（只校验前缀，不校验扩展名，后面用正则判断）
		if (!preg_match('/^https?:\/\//i', $url)) {
			echo json_encode(['success' => false, 'message' => '图片链接格式不正确']);
			exit;
		}

		// 下载图片
		$imgData = @file_get_contents($url);
		if ($imgData === false) {
			echo json_encode(['success' => false, 'message' => '图片下载失败']);
			exit;
		}
		$this->logAction('远程图片下载，图片名: ' . basename($url));

		// 生成保存目录（与上传一致）
		$today = date('Ymd');
		$uploadDir = 'uploads/image/' . $today . '/';
		if (!is_dir($uploadDir)) mkdir($uploadDir, 0777, true);

		// 用正则从 path 部分提取扩展名
		$path = parse_url($url, PHP_URL_PATH);
		if (preg_match('/\.(jpg|jpeg|png|gif|bmp|webp|ico)$/i', $path, $match)) {
			$ext = strtolower($match[1]);
		} else {
			$ext = 'jpg';
		}
		$timestamp = time();
		$rand = mt_rand(100000, 999999);
		$uniqueName = $timestamp . $rand . '.' . $ext;
		$filePath = $uploadDir . $uniqueName;

		// 保存图片
		if (file_put_contents($filePath, $imgData) === false) {
			echo json_encode(['success' => false, 'message' => '图片保存失败']);
			exit;
		}

		// 写入元数据
		$metadataFile = './uploads/metadata.json';
		if (!file_exists($metadataFile)) file_put_contents($metadataFile, json_encode([]));
		$metadata = json_decode(file_get_contents($metadataFile), true);
		if (!is_array($metadata)) $metadata = [];

		// 计算最大批次号
		$maxBatch = 0;
		foreach ($metadata as $item) {
			if (isset($item['batch_number']) && $item['batch_number'] > $maxBatch) {
				$maxBatch = $item['batch_number'];
			}
		}
		$newBatch = $maxBatch + 1;

		$metadata[$uniqueName] = [
			'name' => basename($url),
			'upload_time' => date('Y-m-d H:i:s'),
			'group_id' => 0, // 默认分组
			'batch_number' => $newBatch // 新批次号
		];
		file_put_contents($metadataFile, json_encode($metadata, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT));

		// 更新分组数量
		$groupsFile = './uploads/image_groups.json';
		if (file_exists($groupsFile)) {
			$groups = json_decode(file_get_contents($groupsFile), true);
			if (isset($groups['groups']) && is_array($groups['groups'])) {
				foreach ($groups['groups'] as &$group) {
					if ($group['id'] == 0) {
						$group['count'] = isset($group['count']) ? $group['count'] + 1 : 1;
						break;
					}
				}
				file_put_contents($groupsFile, json_encode($groups, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT));
			}
		}

		// ====== 关键：只更新对应AI消息的img_url和downloaded字段 ======
		$chatHistoryFile = './uploads/chat_history.json';
		if (file_exists($chatHistoryFile)) {
			$history = json_decode(file_get_contents($chatHistoryFile), true);
			$changed = false;
			foreach ($history as &$msg) {
				// 只更新 type=ai 且 origin_url 完全匹配的消息
				if (isset($msg['type']) && $msg['type'] === 'ai' && isset($msg['origin_url']) && $msg['origin_url'] == $url) {
					$msg['img_url'] = '/' . $filePath;
					$msg['downloaded'] = true;
					$changed = true;
				}
			}
			unset($msg);
			if ($changed) {
				file_put_contents($chatHistoryFile, json_encode($history, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT));
			}
		}
		// ====== 关键代码结束 ======

		echo json_encode([
			'success' => true,
			'url' => '/' . $filePath
		]);
	}
	

	/**
	 * 重新计算每个分组包含的图片数量
	*/
	function recountGroupImages(&$data) {
		global $imageDbFile;
		
		// 查找可能的图片数据库文件
		$possibleFiles = [
			'./uploads/metadata.json',
		];
		
		$imagesDb = null;
		foreach ($possibleFiles as $file) {
			if (file_exists($file)) {
				$content = file_get_contents($file);
				$imagesDb = json_decode($content, true);
				if ($imagesDb) {
					break;
				}
			}
		}

		if ($imagesDb === null) {
			return;
		}
		
		// 重置所有分组的计数
		foreach ($data['groups'] as &$group) {
			$group['count'] = 0;
		}
		
		// 重新计算每个分组的图片数量
		$imagesArray = isset($imagesDb['images']) ? $imagesDb['images'] : $imagesDb;
		if (!is_array($imagesArray)) {
			return;
		}
		
		// 尝试不同的字段名称
		$possibleFields = ['groupId', 'group_id', 'groupid', 'group-id', 'group'];
		
		foreach ($imagesArray as $image) {
			$groupId = 0; // 默认分组ID
			
			// 查找图片所属的分组ID
			foreach ($possibleFields as $field) {
				if (isset($image[$field])) {
					$groupId = (int)$image[$field];
					break;
				}
			}
			
			// 增加对应分组的计数
			foreach ($data['groups'] as &$group) {
				if ($group['id'] == $groupId) {
					$group['count']++;
					break;
				}
			}
		}
	}

	/**
	 * 更新所有分组的图片计数
	 */
	function updateAllGroupCounts()
	{
		global $metadataFile, $groupsFile;

		// 检查必要文件是否存在
		if (!file_exists($metadataFile) || !file_exists($groupsFile)) {
			return false;
		}

		// 读取元数据和分组数据
		$metadata = json_decode(file_get_contents($metadataFile), true);
		if (!is_array($metadata)) $metadata = [];
		$groupsData = json_decode(file_get_contents($groupsFile), true) ?: ['groups' => []];

		// 初始化计数器
		$counts = [];
		foreach ($groupsData['groups'] as $group) {
			$counts[$group['id']] = 0;
		}

		// 统计每个分组的图片数量
		foreach ($metadata as $item) {
			// 兼容对象数组和以文件名为key的对象
			if (is_array($item) && isset($item['group_id'])) {
				$groupId = intval($item['group_id']);
			} else {
				$groupId = 0;
			}
			if (isset($counts[$groupId])) {
				$counts[$groupId]++;
			} else {
				$counts[$groupId] = 1;
			}
		}
		
		// 更新分组数据中的计数
		foreach ($groupsData['groups'] as &$group) {
			$groupId = $group['id'];
			$group['count'] = isset($counts[$groupId]) ? $counts[$groupId] : 0;
		}
	
		// 保存更新后的分组数据
		file_put_contents($groupsFile, json_encode($groupsData, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT));
		return true;
	}



}