<?php
declare (strict_types = 1);

namespace app\admin\controller;

use think\Request;
use think\facade\View;
use think\facade\Db;

class Label extends Common
{
    /**
     * 显示资源列表
     *
     * @return \think\Response
     */
    public function index()
    {
        
        // 权限拦截：查看权限
        if (!$this->hasPermission('label_view')) {
            if (request()->isAjax() || request()->isPost()) {
                return json(['status'=>'fail','msg'=>'无权限查看标签']);
            }
            return '<div style="display:flex;align-items:center;justify-content:center;height:80vh;font-size:22px;color:#c00;font-weight:bold;">当前用户无访问标签权限！</div>';
        }
        $lst=Db::name('label')->order('id desc')->select();
        View::assign([
            'lst'=>$lst,
        ]);
        return View::fetch('index');
    }

    public function value()
    {
        // 权限拦截：编辑权限
        if (!$this->hasPermission('label_edit')) {
            return json(['status'=>'fail','msg'=>'无权限编辑标签']);
        }
        $data=input('post.');

        // 获取数据库中的原始数据
        $originalData = Db::name('label')->where('name', 'in', array_keys($data))->column('value', 'name');
        
        // 只更新有变动的定制标签，并记录日志
        $logArr = [];
        $hasChanges = false;
        foreach ($data as $key => $value) {
            $old = isset($originalData[$key]) ? $originalData[$key] : '';
            $old = is_null($old) ? '' : $old;
            $new = is_null($value) ? '' : $value;
            if ($old != $new) {
                $hasChanges = true;
                Db::name('label')->where(['name'=>$key])->update(['value'=>$value]);
                $logArr[] = ['标签名称:'=>$key, '原'=>$old, '改为'=>$new];
            }
        }
        if (!$hasChanges) {
            return json(['status' => 'fail', 'msg' => '没有做任何修改']);
        }
        if (!empty($logArr)) {
            $this->logAction('更新定制标签:' . json_encode($logArr, JSON_UNESCAPED_UNICODE));
        }
        return json(['msg'=>'更新成功','status'=>'success']);
        
    }

    /**
     * 显示创建资源表单页.
     *
     * @return \think\Response
     */

    
    public function create()
    {
        if (!$this->hasPermission('label_add')) {
            if (request()->isAjax() || request()->isPost()) {
                return json(['status'=>'fail','msg'=>'无权限添加']);
            }
            return '<div style="display:flex;align-items:center;justify-content:center;height:80vh;font-size:22px;color:#c00;font-weight:bold;">当前用户无添加权限！</div>';
        }
        return View::fetch('add');
        
        
    }

    public function getlabel(Request $request)
    {
        if (!$this->hasPermission('label_view')) {
            return json(['code'=>0, 'count'=>0, 'data'=>[], 'msg'=>'无权限']);
        }
        $limit = $request->param('limit')?$request->param('limit'):15;
        $builder = Db::name('label');
        $table =$builder->order('id desc')->paginate(['list_rows'=>$limit])->toArray();
        return json(['code'=>0, 'count'=>$table['total'], 'data'=>$table['data']]);
        
    }

    function hasChinese($text) {
        return preg_match('/[\x{4e00}-\x{9fa5}]/u', $text);
    }
    /**
     * 保存新建的资源
     *
     * @param  \think\Request  $request
     * @return \think\Response
     */
    public function save(Request $request)
    {
        // 权限拦截：添加权限
        if (!$this->hasPermission('label_add')) {
            return json(['status'=>'fail','msg'=>'无权限添加标签']);
        }
        $data=input('post.');

        if($data['name'] && ! preg_match('/^[a-zA-Z0-9\-_\/]+$/', $data['name'])) {
            return json(['msg'=>'标签名称只允许字母、数字、横线、斜线组成！','status'=>'fail']);
        }else{
            $label=Db::name('label')->where(['name'=>$data['name']])->find();
            if($label){
                return json(['msg'=>'该自定义标签称已经存在，不能再使用！','status'=>'fail']);
            }else{
                $data['time']=time();
                Db::name('label')->insert($data);
                    $this->logAction('添加定制标签，标签名:' . $data['name']);
                return json(['msg'=>'添加成功','status'=>'success']);
            }
        }
    }

   

    /**
     * 显示编辑资源表单页.
     *
     * @param  int  $id
     * @return \think\Response
     */
    public function edit($id)
    {
       
        // 权限拦截：编辑权限
        if (!$this->hasPermission('label_edit')) {
            if (request()->isAjax() || request()->isPost()) {
                return json(['status'=>'fail','msg'=>'无权限编辑标签']);
            }
            return '<div style="display:flex;align-items:center;justify-content:center;height:80vh;font-size:22px;color:#c00;font-weight:bold;">当前用户无编辑标签权限！</div>';
        }
        $edit=Db::name('label')->find($id);
        View::assign([
            'edit'=>$edit,
        ]);
        return View::fetch('edit');
    }

    /**
     * 保存更新的资源
     *
     * @param  \think\Request  $request
     * @param  int  $id
     * @return \think\Response
     */
    public function update()
    {
        // 权限拦截：编辑权限
        if (!$this->hasPermission('label_edit')) {
            return json(['status'=>'fail','msg'=>'无权限编辑标签']);
        }
        $data=input('post.');
        $id=$data['id'];
        // 获取数据库中的原始数据
        $originalData = Db::name('label')->where('id', $id)->find();

        // 比较数据是否有变化
        $hasChanges = false;
        foreach ($data as $key => $value) {
            if (isset($originalData[$key]) && $originalData[$key] != $value) {
                $hasChanges = true;
                break;
            }
        }
        
        if (!$hasChanges) {
            return json(['status' => 'fail', 'msg' => '没有做任何更新']);
        }
        
        if($data['name'] && ! preg_match('/^[a-zA-Z0-9\-_\/]+$/', $data['name'])) {
            return json(['msg'=>'标签名称只允许字母、数字、横线、斜线组成！','status'=>'fail']);
        }else{
            $label=Db::name('label')->where(['name'=>$data['name']])->find();
            if($label){
                if($label['id']==$data['id']){
                    $label=Db::name('label')->where(['id'=>$data['id']])->save($data);
                        $this->logAction('更新定制标签，ID:' . $data['id'] . '，标签名:' . $data['name']);
                    if($label){
                        return json(['status'=>'success','msg'=>'更新成功']);
                    }else{
                        return json(['status'=>'fail','msg'=>'更新失败']);
                    }
                }else{
                    return json(['status'=>'fail','msg'=>'字段名称已经存在，不能重复添加！']);
                }
            }else{
                $label=Db::name('label')->where(['id'=>$data['id']])->save($data);
                    $this->logAction('更新定制标签，ID:' . $data['id'] . '，标签名:' . $data['name']);
                return json(['status'=>'success','msg'=>'更新成功']);
            }
        }
        
    }

    public function click()
    {
        if (!$this->hasPermission('label_edit')) {
            return json(['status'=>'fail','msg'=>'无权限编辑']);
        }
        $data=input('post.');
            $label = Db::name('label')->where(['id'=>$data['id']])->find();
            Db::name('label')->where(['id'=>$data['id']])->update([$data['field']=>$data['value']]);
            $name = isset($label['name']) ? $label['name'] : '';
            $this->logAction('修改标签字段，ID:' . $data['id'] . '，标签名:' . $name . '，字段:' . $data['field'] . '，值:' . $data['value']);
        
    }

    /**
     * 删除指定资源
     *
     * @param  int  $id
     * @return \think\Response
     */
    public function delete($id)
    {
        // 权限拦截：删除权限
        if (!$this->hasPermission('label_delete')) {
            return json(['status'=>'fail','msg'=>'无权限删除标签']);
        }
            $label = Db::name('label')->where('id', $id)->find();
            $name = isset($label['name']) ? $label['name'] : '';
            $res = Db::name('label')->where('id',$id)->delete();
            if($res){
                $this->logAction('删除定制标签，ID:' . $id . '，标签名:' . $name);
                return json(['status'=>'success','msg'=>'删除成功']);
            }else{
                return json(['status'=>'fail','msg'=>'删除失败']);
            }
    }
    
    public function del()
    {
        // 权限拦截：批量删除权限
        if (!$this->hasPermission('label_delete')) {
            return json(['status'=>'fail','msg'=>'无权限批量删除标签']);
        }
        $data=input('post.');
            $ids = $data['id'];
            $names = [];
            if (is_array($ids)) {
                $labels = Db::name('label')->where('id', 'in', $ids)->column('name', 'id');
                foreach ($ids as $id) {
                    $names[] = isset($labels[$id]) ? $labels[$id] : '';
                }
            } else {
                $label = Db::name('label')->where('id', $ids)->find();
                $names[] = isset($label['name']) ? $label['name'] : '';
            }
            $del= Db::name('label')->where(['id'=>$ids])->delete();
            if($del){
                $this->logAction('批量删除定制标签，ID:' . json_encode($ids) . '，标签名:' . json_encode($names, JSON_UNESCAPED_UNICODE));
                return json(['status'=>'success','msg'=>'批量删除成功']);
            }else{
                return json(['status'=>'fail','msg'=>'批量删除失败']);
            }
    }

}
